
/*
 * The org.opensourcephysics.controls package defines the framework for building
 * user interface controls for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.controls;
import java.text.DecimalFormat;
import java.awt.Frame;

/**
 * AbstractAnimation is a template for simple animations.
 *
 * Implement the doStep method to create an animation.  This method is called from the run method and when
 * the stepAnimation button is pressed.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public abstract class AbstractAnimation implements Animation, Runnable {
   protected Control control;                                           // the model's control
   protected volatile Thread animationThread;
   protected int delayTime = 100;                                       // time between animation steps in milliseconds

   /** Field decimalFormat can be used to display time and other numeric values. */
   protected DecimalFormat decimalFormat = new DecimalFormat("0.00E0"); // default numeric format for messages

   /**
    * Sets the Control for this model and initializes the control's values.
    *
    * @param control
    */
   public void setControl(Control control) {
      this.control = control;

      if(control!=null) {
         resetAnimation(); // sets the control's default values

         if(control instanceof Frame) {
            ((Frame) control).pack();
         }
      }
   }

   /**
    * Gets the Control.
    *
    * @return
    */
   public Control getControl() {
     return control;
   }


   /**
    * Initializes the animation by reading parameters from the control.
    */
   public void initializeAnimation() {
      control.clearMessages();
   }

   /**
    * Does an animation step.
    */
   abstract protected void doStep();

   /**
    * Stops the animation.
    *
    * Sets the animationThread to null and waits for a join.
    */
   public synchronized void stopAnimation() {
      if(animationThread==null) {
         return;
      }

      Thread tempThread = animationThread; // temporary reference
      animationThread = null; // signal the animation to stop

      if(Thread.currentThread()==tempThread) {
         return; // cannot join own thread so return
      }

      if((tempThread!=null)&&(tempThread!=Thread.currentThread())) {
         try {                      // guard against an exception in applet mode
            tempThread.interrupt(); // get out of a sleep state
            tempThread.join(2000);  // wait up to 2 seconds
         } catch(InterruptedException e) {
            // System.out.println("excetpion in stop animation"+e);
         }
      }
   }

   /**
    * Determines if the animation is running.
    *
    * @return boolean
    */
   public final boolean isRunning() {
      return animationThread!=null;
   }

   /**
    * Steps the animation.
    */
   public synchronized void stepAnimation() {
      if(animationThread!=null) {
         stopAnimation();
      }

      doStep();
   }

   /**
    * Starts the animation.
    *
    * Use this method to start a timer or a thread.
    */
   public synchronized void startAnimation() {
      if(animationThread!=null) {
         return; // animation is running
      }

      animationThread = new Thread(this);
      animationThread.setPriority(Thread.MIN_PRIORITY);
      animationThread.setDaemon(true);
      animationThread.start(); // start the animation
   }

   /**
    * Resets the animation to a predefined state.
    */
   public void resetAnimation() {
      if(animationThread!=null) {
         stopAnimation(); // make sure animation is stopped
      }

      control.clearMessages();
   }

   /**
    * Implementation of Runnable interface.  DO NOT access this method directly.
    */
   public void run() {
      long sleepTime = delayTime;
      while(animationThread==Thread.currentThread()) {
         long currentTime = System.currentTimeMillis();
         doStep();
         //adjust the sleep time to try and achieve a constant animation rate
         sleepTime = Math.max(1, delayTime-(System.currentTimeMillis()-currentTime));

         try {
            Thread.sleep(sleepTime);
         } catch(InterruptedException ie) {}
      }
   }

}

