package JSci.swing;

import java.awt.*;
import java.awt.geom.Point2D;
import JSci.awt.*;

/**
* The JGraph2D superclass provides an abstract encapsulation of 2D graphs.
* @version 1.2
* @author Mark Hale
*/
public abstract class JGraph2D extends JDoubleBufferedComponent implements GraphDataListener {
        /**
        * Data model.
        */
        protected Graph2DModel model;
        /**
        * Axis numbering.
        */
        protected boolean numbering=true;
        /**
        * Origin.
        */
        protected Point origin=new Point();
        /**
        * Min and max data points.
        */
        protected float minX,minY,maxX,maxY;
        private boolean autoYExtrema=true;
        /**
        * Series colors.
        */
        protected Color seriesColor[]={Color.black,Color.blue,Color.green,Color.red,Color.yellow,Color.cyan,Color.lightGray,Color.magenta,Color.orange,Color.pink};
        /**
        * Axis scaling.
        */
        private float xScale,yScale;
        /**
        * Axis numbering increment.
        */
        private float xInc,yInc;
        private boolean autoXInc=true,autoYInc=true;
        /**
        * Padding.
        */
        protected final int scalePad=5;
        protected final int axisPad=25;
        protected int leftAxisPad;
        /**
        * Constructs a 2D graph.
        */
        public JGraph2D(Graph2DModel gm) {
                model=gm;
                model.addGraphDataListener(this);
                dataChanged(new GraphDataEvent(model));
        }
        /**
        * Sets the data plotted by this graph to the specified data.
        */
        public final void setModel(Graph2DModel gm) {
                model.removeGraphDataListener(this);
                model=gm;
                model.addGraphDataListener(this);
                dataChanged(new GraphDataEvent(model));
        }
        /**
        * Returns the model used by this graph.
        */
        public final Graph2DModel getModel() {
                return model;
        }
        /**
        * Implementation of GraphDataListener.
        */
        public void dataChanged(GraphDataEvent e) {
                // determine minX and maxX from model
                float tmpX;
                minX=Float.POSITIVE_INFINITY;
                maxX=Float.NEGATIVE_INFINITY;
                model.firstSeries();
                for(int i=0;i<model.seriesLength();i++) {
                        tmpX=model.getXCoord(i);
                        minX=Math.min(tmpX,minX);
                        maxX=Math.max(tmpX,maxX);
                }
                int n;
                for(n=1;model.nextSeries();n++) {
                        for(int i=0;i<model.seriesLength();i++) {
                                tmpX=model.getXCoord(i);
                                minX=Math.min(tmpX,minX);
                                maxX=Math.max(tmpX,maxX);
                        }
                }
                if(minX==Float.POSITIVE_INFINITY || maxX==Float.NEGATIVE_INFINITY) {
                        minX=-5.0f;
                        maxX=5.0f;
                }
                // ensure there are enough colors
                if(n>seriesColor.length) {
                        Color tmp[]=seriesColor;
                        seriesColor=new Color[n];
                        System.arraycopy(tmp,0,seriesColor,0,tmp.length);
                        for(int i=tmp.length;i<n;i++)
                                seriesColor[i]=seriesColor[i-tmp.length];
                }
                if(autoYExtrema)
                        setYExtrema(0.0f,0.0f);
                setNumbering(numbering);
                rescale();
        }
        /**
        * Turns axis numbering on/off.
        * Default is on.
        */
        public final void setNumbering(boolean flag) {
                numbering=flag;
                leftAxisPad=axisPad;
                if(numbering) {
                        int yNumPad=8*Math.max(String.valueOf(maxY).length(),String.valueOf(minY).length());
                        if(minX<0.0f) {
                                // Swing optimised
                                final int negXLen=(int)((Math.max(getWidth(),getMinimumSize().width)-2*(axisPad+scalePad))*minX/(minX-maxX));
                                yNumPad=Math.max(yNumPad-negXLen,0);
                        }
                        leftAxisPad+=yNumPad;
                }
        }
        /**
        * Sets the x-axis numbering increment.
        * @param dx use 0.0f for auto-adjusting (default).
        */
        public final void setXIncrement(float dx) {
                if(dx<0.0f)
                        throw new IllegalArgumentException("Increment should be positive.");
                else if(dx==0.0f)
                        autoXInc=true;
                else
                        autoXInc=false;
                xInc=dx;
        }
        /**
        * Sets the y-axis numbering increment.
        * @param dy use 0.0f for auto-adjusting (default).
        */
        public final void setYIncrement(float dy) {
                if(dy<0.0f)
                        throw new IllegalArgumentException("Increment should be positive.");
                else if(dy==0.0f)
                        autoYInc=true;
                else
                        autoYInc=false;
                yInc=dy;
        }
        /**
        * Sets the minimum/maximum values on the y-axis.
        * Set both min and max to 0.0f for auto-adjusting (default).
        */
        public final void setYExtrema(float min,float max) {
                if(min==0.0f && max==0.0f) {
                        autoYExtrema=true;
                        // determine minY and maxY from model
                        float tmp;
                        minY=Float.POSITIVE_INFINITY;
                        maxY=Float.NEGATIVE_INFINITY;
                        model.firstSeries();
                        for(int i=0;i<model.seriesLength();i++) {
                                tmp=model.getYCoord(i);
                                minY=Math.min(tmp,minY);
                                maxY=Math.max(tmp,maxY);
                        }
                        while(model.nextSeries()) {
                                for(int i=0;i<model.seriesLength();i++) {
                                        tmp=model.getYCoord(i);
                                        minY=Math.min(tmp,minY);
                                        maxY=Math.max(tmp,maxY);
                                }
                        }
                        if(minY==maxY) {
                                minY-=0.5f;
                                maxY+=0.5f;
                        }
                        if(minY==Float.POSITIVE_INFINITY || maxY==Float.NEGATIVE_INFINITY) {
                                minY=-5.0f;
                                maxY=5.0f;
                        }
                } else if(max<=min) {
                        throw new IllegalArgumentException("Maximum should be greater than minimum; max = "+max+" and min = "+min);
                } else {
                        autoYExtrema=false;
                        minY=min;
                        maxY=max;
                }
                rescale();
        }
        /**
        * Sets the color of the nth y-series.
        * @param n the index of the y-series.
        * @param c the line color.
        */
        public final void setColor(int n,Color c) {
                seriesColor[n]=c;
        }
        /**
        * Reshapes the JGraph2D to the specified bounding box.
        */
        public final void setBounds(int x,int y,int width,int height) {
                super.setBounds(x,y,width,height);
                rescale();
        }
        /**
        * Returns the preferred size of this component.
        */
        public Dimension getPreferredSize() {
                return getMinimumSize();
        }
        /**
        * Returns the minimum size of this component.
        */
        public Dimension getMinimumSize() {
                return new Dimension(200,200);
        }
        /**
        * Rescales the JGraph2D.
        */
        protected final void rescale() {
                final Dimension s=getMinimumSize();
                // Swing optimised
                final int thisWidth=Math.max(getWidth(),s.width);
                final int thisHeight=Math.max(getHeight(),s.height);
                xScale=(thisWidth-(leftAxisPad+axisPad))/(maxX-minX);
                yScale=(thisHeight-2*axisPad)/(maxY-minY);
                if(autoXInc)
                        xInc=round(40.0f/xScale);
                if(autoYInc)
                        yInc=round(40.0f/yScale);
                origin.x=leftAxisPad-Math.round(minX*xScale);
                origin.y=thisHeight-axisPad+Math.round(minY*yScale);
                redraw();
        }
        /**
        * Converts a data point to screen coordinates.
        */
        protected final Point dataToScreen(float x,float y) {
                return new Point(origin.x+Math.round(xScale*x),origin.y-Math.round(yScale*y));
        }
        /**
        * Converts a screen point to data coordinates.
        */
        protected final Point2D.Float screenToData(Point p) {
                return new Point2D.Float((p.x-origin.x)/xScale,(origin.y-p.y)/yScale);
        }
        /**
        * Draws the graph axes.
        */
        protected final void drawAxes(Graphics g) {
// axis - Swing optimised
                g.setColor(getForeground());
                if(minY>0.0f)
                        g.drawLine(leftAxisPad-scalePad,getHeight()-axisPad,getWidth()-(axisPad-scalePad),getHeight()-axisPad);
                else
                        g.drawLine(leftAxisPad-scalePad,origin.y,getWidth()-(axisPad-scalePad),origin.y);
                if(minX>0.0f)
                        g.drawLine(leftAxisPad,axisPad-scalePad,leftAxisPad,getHeight()-(axisPad-scalePad));
                else
                        g.drawLine(origin.x,axisPad-scalePad,origin.x,getHeight()-(axisPad-scalePad));
// numbering
                if(numbering) {
                        String str;
                        int strWidth;
                        final FontMetrics metrics=g.getFontMetrics();
                        final int strHeight=metrics.getHeight();
                        Point p;
// x-axis numbering
                        float x;
                        for(x=(minX>0.0f)?minX:xInc;x<=maxX;x+=xInc) {
                                str=String.valueOf(round(x));
// add a + prefix to compensate for - prefix in negative number strings when calculating length
                                strWidth=metrics.stringWidth('+'+str);
                                p=dataToScreen(x,(minY>0.0f)?minY:0.0f);
                                g.drawLine(p.x,p.y,p.x,p.y+5);
                                g.drawString(str,p.x-strWidth/2,p.y+5+strHeight);
                        }
                        for(x=-xInc;x>=minX;x-=xInc) {
                                str=String.valueOf(round(x));
                                strWidth=metrics.stringWidth(str);
                                p=dataToScreen(x,(minY>0.0f)?minY:0.0f);
                                g.drawLine(p.x,p.y,p.x,p.y+5);
                                g.drawString(str,p.x-strWidth/2,p.y+5+strHeight);
                        }
// y-axis numbering
                        float y;
                        for(y=(minY>0.0f)?minY:yInc;y<=maxY;y+=yInc) {
                                str=String.valueOf(round(y));
                                strWidth=metrics.stringWidth(str);
                                p=dataToScreen((minX>0.0f)?minX:0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(str,p.x-8-strWidth,p.y+strHeight/4);
                        }
                        for(y=-yInc;y>=minY;y-=yInc) {
                                str=String.valueOf(round(y));
                                strWidth=metrics.stringWidth(str);
                                p=dataToScreen((minX>0.0f)?minX:0.0f,y);
                                g.drawLine(p.x,p.y,p.x-5,p.y);
                                g.drawString(str,p.x-8-strWidth,p.y+strHeight/4);
                        }
                }
        }
        /**
        * Rounds numbers to so many significant figures.
        */
        protected final static float round(float x) {
                final int SIG_FIG=2;
                int sign=1;
                if(x<0.0f)
                        sign=-1;
                final float mag=Math.abs(x);
                int places;
                float tmp,factor;
                if(mag<1.0f) {
                        tmp=10.0f*mag;
                        for(places=1;tmp<1.0f;places++)
                                tmp*=10.0f;
                        factor=(float)Math.pow(10.0f,places+SIG_FIG-1);
                } else {
                        tmp=mag/10.0f;
                        for(places=1;tmp>1.0f;places++)
                                tmp/=10.0f;
                        factor=(float)Math.pow(10.0f,SIG_FIG-places);
                }
                return (sign*Math.round(mag*factor))/factor;
        }
}

