package JSci.maths.categories;

import JSci.maths.*;
import JSci.maths.algebras.*;
import JSci.maths.fields.*;
import JSci.maths.groups.AbelianGroupMember;

/**
* The Hilb class encapsulates the category <b>Hilb</b>.
* @version 1.0
* @author Mark Hale
*/
public class Hilb extends Object implements Category {
        /**
        * Constructs a <b>Hilb</b> category.
        */
        public Hilb() {}
        /**
        * Returns the identity morphism for an object.
        * @param a a HilbertSpace.
        */
        public Morphism identity(Object a) {
                return new LinearMap(ComplexDiagonalMatrix.identity(((HilbertSpace)a).dimension()));
        }
        /**
        * Returns the cardinality of an object.
        * @param a a HilbertSpace.
        */
        public Object cardinality(Object a) {
                return new MathInteger(((HilbertSpace)a).dimension());
        }
        /**
        * Returns a hom-set.
        * @param a a HilbertSpace.
        * @param b a HilbertSpace.
        * @return a HilbertSpace.
        */
        public Homset homset(Object a,Object b) {
                return new OperatorSpace((HilbertSpace)a,(HilbertSpace)b);
        }
        public class OperatorSpace extends HilbertSpace implements Homset {
                private final int rows,cols;
                public OperatorSpace(HilbertSpace a,HilbertSpace b) {
                        super(a.dimension()*b.dimension());
                        rows=b.dimension();
                        cols=a.dimension();
                }
                /**
                * Returns an element of this hom-set.
                */
                public VectorSpaceMember getVector(Complex array[][]) {
                        return new LinearMap(array);
                }
        }
        public class LinearMap implements BanachSpaceMember, Morphism {
                private ComplexMatrix matrix;
                public LinearMap(Complex array[][]) {
                        matrix=new ComplexMatrix(array);
                }
                public LinearMap(ComplexMatrix m) {
                        matrix=m;
                }
                public Object domain() {
                        return new HilbertSpace(matrix.columns());
                }
                public Object codomain() {
                        return new HilbertSpace(matrix.rows());
                }
                public Object map(Object v) {
                        return matrix.multiply((ComplexVector)v);
                }
                public Morphism compose(Morphism m) {
                        if(m instanceof LinearMap) {
                                LinearMap lm=(LinearMap)m;
                                if(matrix.columns()==lm.matrix.rows())
                                        return new LinearMap(lm.matrix.multiply(matrix));
                                else
                                        throw new UndefinedCompositionException();
                        } else
                                throw new IllegalArgumentException("Morphism is not a LinearMap.");
                }
                public double norm() {
                        return matrix.frobeniusNorm();
                }
                public int dimension() {
                        return matrix.rows()*matrix.columns();
                }
                public AbelianGroupMember add(final AbelianGroupMember m) {
                        if(m instanceof LinearMap)
                                return new LinearMap(matrix.add(((LinearMap)m).matrix));
                        else
                                throw new IllegalArgumentException("Member class not recognised by this method.");
                }
                public AbelianGroupMember negate() {
                        return new LinearMap((ComplexMatrix)matrix.negate());
                }
                public AbelianGroupMember subtract(final AbelianGroupMember m) {
                        if(m instanceof LinearMap)
                                return new LinearMap(matrix.subtract(((LinearMap)m).matrix));
                        else
                                throw new IllegalArgumentException("Member class not recognised by this method.");
                }
                public ModuleMember scalarMultiply(RingMember z) {
                        if(z instanceof Complex)
                                return new LinearMap(matrix.scalarMultiply((Complex)z));
                        else
                                throw new IllegalArgumentException("Member class not recognised by this method.");
                }
                public VectorSpaceMember scalarDivide(FieldMember z) {
                        if(z instanceof Complex)
                                return new LinearMap(matrix.scalarMultiply((Complex)z));
                        else
                                throw new IllegalArgumentException("Member class not recognised by this method.");
                }
        }
}

